import { storage } from './storage.js';

export const theme = (() => {

    const THEME_DARK = 'dark';
    const themeColors = {
        '#000000': '#FFFFFF',
        '#FFFFFF': '#000000',
        '#212529': '#F8F9FA',
        '#F8F9FA': '#212529'
    };

    let theme = null;
    let isAuto = false;
    let observerLight = null;
    let observerDark = null;

    const toDark = (element) => {
        if (element.classList.contains('text-dark')) {
            element.classList.remove('text-dark');
            element.classList.add('text-light');
        }

        if (element.classList.contains('btn-theme-dark')) {
            element.classList.remove('btn-theme-dark');
            element.classList.add('btn-theme-light');
        }

        if (element.classList.contains('bg-light')) {
            element.classList.remove('bg-light');
            element.classList.add('bg-dark');
        }

        if (element.classList.contains('bg-white')) {
            element.classList.remove('bg-white');
            element.classList.add('bg-black');
        }

        if (element.classList.contains('bg-theme-light')) {
            element.classList.remove('bg-theme-light');
            element.classList.add('bg-theme-dark');
        }

        if (element.classList.contains('color-theme-white')) {
            element.classList.remove('color-theme-white');
            element.classList.add('color-theme-black');
        }

        if (element.classList.contains('btn-outline-dark')) {
            element.classList.remove('btn-outline-dark');
            element.classList.add('btn-outline-light');
        }

        if (element.classList.contains('bg-cover-white')) {
            element.classList.remove('bg-cover-white');
            element.classList.add('bg-cover-black');
        }
    };

    const onDark = () => {
        theme.set('active', THEME_DARK);
        document.documentElement.setAttribute('data-bs-theme', THEME_DARK);

        const now = document.querySelector('meta[name="theme-color"]').getAttribute('content');
        const elements = document.querySelectorAll('.text-dark, .btn-theme-dark, .bg-light, .bg-white, .bg-theme-light, .color-theme-white, .btn-outline-dark, .bg-cover-white');

        let countChange = 0;
        elements.forEach((el) => {
            const callback = (e) => {
                if (el.isEqualNode(e.target) && (e.propertyName === 'background-color' || e.propertyName === 'color')) {
                    countChange += 1;

                    if (elements.length === countChange) {
                        document.querySelector('meta[name="theme-color"]').setAttribute('content', (now === '#FFFFFF' || now === '#F8F9FA') ? themeColors[now] : now);
                    }
                }
            };

            el.removeEventListener('transitionend', callback);
            el.addEventListener('transitionend', callback);
        });

        elements.forEach((el) => {
            observerDark.observe(el);
        });
    };

    const isDarkMode = (onDark = null, onLight = null) => {
        const status = theme.get('active') === THEME_DARK;

        if (onDark && onLight) {
            return status ? onDark : onLight;
        }

        return status;
    };

    const change = () => {
        if (isDarkMode()) {
            onLight();
        } else {
            onDark();
        }
    };

    const showButtonChangeTheme = () => {
        if (!isAuto) {
            return;
        }

        document.getElementById('button-theme').style.display = 'block';
    };

    const spyTop = () => {
        const observerTop = new IntersectionObserver((es) => {
            es.forEach((e) => {
                if (e.isIntersecting) {
                    const themeColor = ['bg-black', 'bg-white'].some((i) => e.target.classList.contains(i))
                        ? isDarkMode('#000000', '#FFFFFF')
                        : isDarkMode('#212529', '#F8F9FA');

                    document.querySelector('meta[name="theme-color"]').setAttribute('content', themeColor);
                }
            });
        }, {
            rootMargin: '0% 0% -95% 0%',
        });

        document.querySelectorAll('section').forEach((el) => {
            observerTop.observe(el);
        });
    };

    const init = () => {
        theme = storage('theme');

        observerLight = new IntersectionObserver((es, o) => {
            es.forEach((e) => {
                if (e.isIntersecting) {
                    toLight(e.target);
                }
            });

            es.forEach((e) => {
                if (!e.isIntersecting) {
                    toLight(e.target);
                }
            });

            o.disconnect();
        });

        observerDark = new IntersectionObserver((es, o) => {
            es.forEach((e) => {
                if (e.isIntersecting) {
                    toDark(e.target);
                }
            });

            es.forEach((e) => {
                if (!e.isIntersecting) {
                    toDark(e.target);
                }
            });

            o.disconnect();
        });

        if (!theme.has('active')) {
            theme.set('active', THEME_LIGHT);

            if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
                theme.set('active', THEME_DARK);
            }
        }

        switch (document.body.getAttribute('data-theme')) {
            case 'dark':
                theme.set('active', THEME_DARK);
                break;
            case 'light':
                theme.set('active', THEME_LIGHT);
                break;
            default:
                isAuto = true;
                break;
        }

        if (isDarkMode()) {
            onDark();
        } else {
            onLight();
        }

        const toggle = document.getElementById('darkMode');
        if (toggle) {
            toggle.checked = isDarkMode();

            if (!isAuto) {
                toggle.parentElement.remove();
            }
        }
    };

    return {
        change,
        init,
        spyTop,
        isDarkMode,
        showButtonChangeTheme
    };
})();